/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%           RRRR    EEEEE   SSSSS   OOO   U   U  RRRR    CCCC  EEEEE          %
%           R   R   E       SS     O   O  U   U  R   R  C      E              %
%           RRRR    EEE      SSS   O   O  U   U  RRRR   C      EEE            %
%           R R     E          SS  O   O  U   U  R R    C      E              %
%           R  R    EEEEE   SSSSS   OOO    UUU   R  R    CCCC  EEEEE          %
%                                                                             %
%                                                                             %
%                        Get/Set ImageMagick Resources.                       %
%                                                                             %
%                                                                             %
%                              Software Design                                %
%                                John Cristy                                  %
%                               September 2002                                %
%                                                                             %
%                                                                             %
%  Copyright (C) 2003 ImageMagick Studio, a non-profit organization dedicated %
%  to making software imaging solutions freely available.                     %
%                                                                             %
%  Permission is hereby granted, free of charge, to any person obtaining a    %
%  copy of this software and associated documentation files ("ImageMagick"),  %
%  to deal in ImageMagick without restriction, including without limitation   %
%  the rights to use, copy, modify, merge, publish, distribute, sublicense,   %
%  and/or sell copies of ImageMagick, and to permit persons to whom the       %
%  ImageMagick is furnished to do so, subject to the following conditions:    %
%                                                                             %
%  The above copyright notice and this permission notice shall be included in %
%  all copies or substantial portions of ImageMagick.                         %
%                                                                             %
%  The software is provided "as is", without warranty of any kind, express or %
%  implied, including but not limited to the warranties of merchantability,   %
%  fitness for a particular purpose and noninfringement.  In no event shall   %
%  ImageMagick Studio be liable for any claim, damages or other liability,    %
%  whether in an action of contract, tort or otherwise, arising from, out of  %
%  or in connection with ImageMagick or the use or other dealings in          %
%  ImageMagick.                                                               %
%                                                                             %
%  Except as contained in this notice, the name of the ImageMagick Studio     %
%  shall not be used in advertising or otherwise to promote the sale, use or  %
%  other dealings in ImageMagick without prior written authorization from the %
%  ImageMagick Studio.                                                        %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%
*/

/*
  Include declarations.
*/
#include "magick/studio.h"
#include "magick/error.h"
#include "magick/log.h"
#include "magick/resource.h"
#include "magick/semaphore.h"
#include "magick/signature.h"
#include "magick/utility.h"

/*
  Define  declarations.
*/
#define ResourceInfinity  (~0UL)
#define ResourceToMegabytes(value) ((double) (value)*1024.0*1024.0)
#define MegabytesToResource(value) ((unsigned long) ((value)/1024.0/1024.0))
#define GigabytesToResource(value) \
  ((unsigned long) ((value)/1024.0/1024.0/1024.0))

/*
  Typedef declarations.
*/
typedef struct _ResourceInfo
{
  double
    file,
    memory,
    map,
    disk;

  unsigned long
    file_limit,
    memory_limit,
    map_limit,
    disk_limit;
} ResourceInfo;

typedef struct _UniqueFileResourceInfo
{
  const char
    *name;

  struct _UniqueFileResourceInfo
    *next,
    *previous;
} UniqueFileResourceInfo;


/*
  Global declarations.
*/
static SemaphoreInfo
  *resource_semaphore = (SemaphoreInfo *) NULL;

static SemaphoreInfo
  *unique_semaphore = (SemaphoreInfo *) NULL;

static ResourceInfo
  resource_info =
  {
    0, 0, 0, 0, 1024, 1024, 4096, ResourceInfinity
  };

static UniqueFileResourceInfo
  *temporary_resources = (UniqueFileResourceInfo *) NULL;

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   A c q u i r e M a g i c k R e s o u r c e                                 %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  AcquireMagickResource() acquires resources of the specified type.  True is
%  returned if the specified resource is available otherwise False.
%
%  The format of the AcquireMagickResource() method is:
%
%      unsigned int AcquireMagickResource(const ResourceType type,
%        const ExtendedSignedIntegralType size)
%
%  A description of each parameter follows:
%
%    o type: The type of resource.
%
%    o size: The number of bytes needed from for this resource.
%
%
*/
MagickExport unsigned int AcquireMagickResource(const ResourceType type,
  const ExtendedSignedIntegralType size)
{
  char
    message[MaxTextExtent];

  unsigned int
    status;

  status=True;
  AcquireSemaphoreInfo(&resource_semaphore);
  switch (type)
  {
    case FileResource:
    {
      resource_info.file+=size;
      if (resource_info.file_limit == ResourceInfinity)
        break;
      status=resource_info.file <= resource_info.file_limit;
      FormatString(message,"file +%lu/%lu/%lu",(unsigned long) size,
        (unsigned long) resource_info.file,resource_info.file_limit);
      (void) LogMagickEvent(ResourceEvent,GetMagickModule(),message);
      break;
    }
    case MemoryResource:
    {
      resource_info.memory+=size;
      if (resource_info.memory_limit == ResourceInfinity)
        break;
      status=resource_info.memory <=
        ResourceToMegabytes(resource_info.memory_limit);
      FormatString(message,"memory +%lumb/%lumb/%lumb",
        MegabytesToResource(size),MegabytesToResource(resource_info.memory),
        resource_info.memory_limit);
      (void) LogMagickEvent(ResourceEvent,GetMagickModule(),message);
      break;
    }
    case MapResource:
    {
      resource_info.map+=size;
      if (resource_info.map_limit == ResourceInfinity)
        break;
      status=resource_info.disk <=
        ResourceToMegabytes(resource_info.map_limit);
      FormatString(message,"map +%lumb/%lumb/%lumb",MegabytesToResource(size),
        MegabytesToResource(resource_info.map),resource_info.map_limit);
      (void) LogMagickEvent(ResourceEvent,GetMagickModule(),message);
      break;
    }
    case DiskResource:
    {
      resource_info.disk+=size;
      if (resource_info.disk_limit == ResourceInfinity)
        break;
      status=resource_info.disk <=
        ResourceToMegabytes(resource_info.disk_limit);
      FormatString(message,"disk +%lumb/%lugb/%lugb",MegabytesToResource(size),
        GigabytesToResource(resource_info.disk),resource_info.disk_limit/1024);
      (void) LogMagickEvent(ResourceEvent,GetMagickModule(),message);
      break;
    }
    default:
      break;
  }
  LiberateSemaphoreInfo(&resource_semaphore);
  return(status);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   A c q u i r e U n i q u e F i l e R e s o u r c e                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  AcquireUniqueFileResource() returns a unique file name, and returns a file
%  descriptor for the file open for reading and writing.
%
%  The format of the AcquireUniqueFileResource() method is:
%
%      int AcquireUniqueFileResource(char *path)
%
%  A description of each parameter follows:
%
%   o  path:  Specifies a pointer to an array of characters.  The unique path
%      name is returned in this array.
%
%
*/

static unsigned int GetPathTemplate(char *path)
{
  char
    *directory;

  int
    status;

  struct stat
    file_info;

  (void) strcpy(path,"magicXXXXXX");
  directory=getenv("TMPDIR");
#if defined(WIN32)
  if (directory == (char *) NULL)
    directory=getenv("TMP");
  if (directory == (char *) NULL)
    directory=getenv("TEMP");
#endif
#if defined(P_tmpdir)
  if (directory == (char *) NULL)
    directory=P_tmpdir;
#endif
  if (directory == (char *) NULL)
    return(True);
  if (strlen(directory) > (MaxTextExtent-15))
    return(True);
  status=stat(directory,&file_info);
  if ((status != 0) || !S_ISDIR(file_info.st_mode))
    return(True);
  if (directory[strlen(directory)-1] == *DirectorySeparator)
    FormatString(path,"%smagicXXXXXX",directory);
  else
    FormatString(path,"%s%smagicXXXXXX",directory,DirectorySeparator);
  return(True);
}

MagickExport int AcquireUniqueFileResource(char *path)
{
#if !defined(O_NOFOLLOW)
#define O_NOFOLLOW 0
#endif
#if !defined(TMP_MAX)
# define TMP_MAX  238328
#endif

  int
    c,
    file;

  register char
    *p;

  register long
    i;

  unsigned char
    key[6];

  assert(path != (char *) NULL);
  file=(-1);
  for (i=0; i < TMP_MAX; i++)
  {
    /*
      Get temporary pathname.
    */
    (void) GetPathTemplate(path);
#if defined(HAVE_MKSTEMP)
    file=mkstemp(path);
    if (file != -1)
      break;
#endif
    GetRandomKey(key,6);
    p=path+strlen(path)-6;
    for (i=0; i < 6; i++)
    {
      c=key[i] & 0x1f;
      *p++=c > 9 ? c+'a'-10 : c+'0';
    }
    file=open(path,O_RDWR | O_CREAT | O_EXCL | O_BINARY | O_NOFOLLOW,S_MODE);
    if ((file > 0) || (errno != EEXIST))
      break;
  }
  if (file != -1)
    {
      UniqueFileResourceInfo
        *resource_info;

      if (!AcquireMagickResource(FileResource,1))
        {
          (void) close(file);
          (void) remove(path);
          return(-1);
        }
      AcquireSemaphoreInfo(&unique_semaphore);
      resource_info=(UniqueFileResourceInfo *)
        AcquireMemory(sizeof(UniqueFileResourceInfo));
      if (resource_info == (UniqueFileResourceInfo *) NULL)
        MagickFatalError(ResourceLimitFatalError,"MemoryAllocationFailed",
          "UnableToAcquireString");
      resource_info->name=AcquireString(path);
      resource_info->next=(UniqueFileResourceInfo *) NULL;
      resource_info->previous=(UniqueFileResourceInfo *) NULL;
      if (temporary_resources == (UniqueFileResourceInfo *) NULL)
        temporary_resources=resource_info;
      else
        {
          resource_info->next=temporary_resources;
          temporary_resources->previous=resource_info;
          temporary_resources=temporary_resources->previous;
        }
      LiberateSemaphoreInfo(&unique_semaphore);
    }
  (void) LogMagickEvent(ResourceEvent,GetMagickModule(),path);
  return(file);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
+   D e s t r o y M a g i c k R e s o u r c e s                               %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  DestroyMagickResources() destroys the resource environment.
%
%  The format of the DestroyMagickResources() method is:
%
%      DestroyMagickResources(void)
%
%
*/
MagickExport void DestroyMagickResources(void)
{
  UniqueFileResourceInfo
    *resource_info;

  register UniqueFileResourceInfo
    *p;

  AcquireSemaphoreInfo(&unique_semaphore);
  for (p=temporary_resources; p != (UniqueFileResourceInfo *) NULL; )
  {
    resource_info=p;
    p=p->next;
    if (resource_info->name != (char *) NULL)
      {
        (void) LogMagickEvent(ResourceEvent,GetMagickModule(),
          resource_info->name);
        (void) remove(resource_info->name);
        LiberateMemory((void **) &resource_info->name);
      }
    LiberateMemory((void **) &resource_info);
  }
  temporary_resources=(UniqueFileResourceInfo *) NULL;
  DestroySemaphoreInfo(&unique_semaphore);
  AcquireSemaphoreInfo(&resource_semaphore);
  DestroySemaphoreInfo(&resource_semaphore);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   G e t M a g i c k R e s o u r c e                                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  GetMagickResource() returns the the specified resource in megabytes.
%
%  The format of the GetMagickResource() method is:
%
%      unsigned long GetMagickResource(const ResourceType type)
%
%  A description of each parameter follows:
%
%    o type: The type of resource.
%
%
*/
MagickExport unsigned long GetMagickResource(const ResourceType type)
{
  unsigned long
    resource;

  resource=0;
  AcquireSemaphoreInfo(&resource_semaphore);
  switch (type)
  {
    case FileResource:
    {
      resource=(unsigned long) resource_info.file;
      break;
    }
    case MemoryResource:
    {
      resource=MegabytesToResource(resource_info.memory);
      break;
    }
    case MapResource:
    {
      resource=MegabytesToResource(resource_info.map);
      break;
    }
    case DiskResource:
    {
      resource=MegabytesToResource(resource_info.disk);
      break;
    }
    default:
      break;
  }
  LiberateSemaphoreInfo(&resource_semaphore);
  return(resource);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
+   I n i t i a l i z e M a g i c k R e s o u r c e s                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  InitializeMagickResources() initializes the resource environment.
%
%  The format of the InitializeMagickResources() method is:
%
%      InitializeMagickResources(void)
%
%
*/
MagickExport void InitializeMagickResources(void)
{
  const char
    *p;

  long
    files,
    limit,
    pagesize,
    pages;

  /*
    Set Magick resource limits.
  */
  p=getenv("MAGICK_DISK_LIMIT");
  if (p != (const char *) NULL)
    SetMagickResourceLimit(DiskResource,atol(p));
  files=0;
#if defined(HAVE_SYSCONF) && defined(_SC_OPEN_MAX)
  files=sysconf(_SC_OPEN_MAX);
#endif
  pagesize=0;
#if defined(HAVE_SYSCONF) && defined(_SC_PAGE_SIZE)
  pagesize=sysconf(_SC_PAGE_SIZE);
#endif
#if defined(HAVE_GETPAGESIZE)
  pagesize=getpagesize();
#endif
  pages=0;
#if defined(HAVE_SYSCONF) && defined(_SC_PHYS_PAGES)
  pages=sysconf(_SC_PHYS_PAGES);
#endif
  limit=((pages+512)/1024)*((pagesize+512)/1024);
#if defined(PixelCacheThreshold)
  limit=PixelCacheThreshold;
#endif
  SetMagickResourceLimit(FileResource,files > 0 ? 3*files/4 : 1024);
  p=getenv("MAGICK_FILES_LIMIT");
  if (p != (const char *) NULL)
    SetMagickResourceLimit(FileResource,atol(p));
  SetMagickResourceLimit(MemoryResource,limit > 0 ? 2*limit : 1024);
  p=getenv("MAGICK_MEMORY_LIMIT");
  if (p != (const char *) NULL)
    SetMagickResourceLimit(MemoryResource,atol(p));
  SetMagickResourceLimit(MapResource,limit > 0 ? 8*limit : 4096);
  p=getenv("MAGICK_MAP_LIMIT");
  if (p != (const char *) NULL)
    SetMagickResourceLimit(MapResource,atol(p));
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   L i b e r a t e M a g i c k R e s o u r c e                               %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  LiberateMagickResource() liberates resources of the specified type.
%
%  The format of the LiberateMagickResource() method is:
%
%      void LiberateMagickResource(const ResourceType type,
%        const ExtendedSignedIntegralType size)
%
%  A description of each parameter follows:
%
%    o type: The type of resource.
%
%    o size: The size of the resource.
%
%
*/
MagickExport void LiberateMagickResource(const ResourceType type,
  const ExtendedSignedIntegralType size)
{
  char
    message[MaxTextExtent];

  AcquireSemaphoreInfo(&resource_semaphore);
  switch (type)
  {
    case FileResource:
    {
      resource_info.file-=size;
      FormatString(message,"file -%lu/%lu/%lu",(unsigned long) size,
        (unsigned long) resource_info.file,resource_info.file_limit);
      (void) LogMagickEvent(ResourceEvent,GetMagickModule(),message);
      break;
    }
    case MemoryResource:
    {
      resource_info.memory-=size;
      FormatString(message,"memory -%lumb/%lumb/%lumb",
        MegabytesToResource(size),MegabytesToResource(resource_info.memory),
        resource_info.memory_limit);
      (void) LogMagickEvent(ResourceEvent,GetMagickModule(),message);
      break;
    }
    case MapResource:
    {
      resource_info.map-=size;
      FormatString(message,"map -%lumb/%lumb/%lumb",MegabytesToResource(size),
        MegabytesToResource(resource_info.map),resource_info.map_limit);
      (void) LogMagickEvent(ResourceEvent,GetMagickModule(),message);
      break;
    }
    case DiskResource:
    {
      resource_info.disk-=size;
      FormatString(message,"disk -%lumb/%lugb/%lugb",MegabytesToResource(size),
        GigabytesToResource(resource_info.disk),resource_info.disk_limit/1024);
      (void) LogMagickEvent(ResourceEvent,GetMagickModule(),message);
      break;
    }
    default:
      break;
  }
  LiberateSemaphoreInfo(&resource_semaphore);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   L i b e r a t e U n i q u e F i l e R e s o u r c e                       %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  (void) LiberateUniqueFileResource() deletes a unique file resource.
%
%  The format of the (void) LiberateUniqueFileResource() method is:
%
%      unsigned int (void) LiberateUniqueFileResource(char *name)
%
%  A description of each parameter follows:
%
%    o name: the name of the temporary resource.
%
%
*/
MagickExport unsigned int LiberateUniqueFileResource(const char *name)
{
  register UniqueFileResourceInfo
    *p;

  (void) LogMagickEvent(ResourceEvent,GetMagickModule(),name);
  AcquireSemaphoreInfo(&unique_semaphore);
  for (p=temporary_resources; p != (UniqueFileResourceInfo *) NULL; p=p->next)
    if (strcmp(p->name,name) == 0)
      {
        UniqueFileResourceInfo
          *resource_info;

        if ((p->previous == (UniqueFileResourceInfo *) NULL) &&
            (p->next == (UniqueFileResourceInfo *) NULL))
          temporary_resources=(UniqueFileResourceInfo *) NULL;
        else
          {
            if (p->previous != (UniqueFileResourceInfo *) NULL)
              {
                p->previous->next=p->next;
                temporary_resources=p->previous;
              }
            if (p->next != (UniqueFileResourceInfo *) NULL)
              {
                p->next->previous=p->previous;
                temporary_resources=p->next;
              }
          }
        resource_info=p;
        LiberateMemory((void **) &resource_info->name);
        LiberateMemory((void **) &resource_info);
        LiberateMagickResource(FileResource,1);
        break;
      }
  (void) remove(name);
  LiberateSemaphoreInfo(&unique_semaphore);
  return(p == (UniqueFileResourceInfo *) NULL);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%  L i s t M a g i c k R e s o u r c e I n f o                                %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  Method ListMagickResourceInfo lists the resource info to a file.
%
%  The format of the ListMagickResourceInfo method is:
%
%      unsigned int ListMagickResourceInfo(FILE *file,ExceptionInfo *exception)
%
%  A description of each parameter follows.
%
%    o file:  An pointer to a FILE.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/
MagickExport unsigned int ListMagickResourceInfo(FILE *file,
  ExceptionInfo *exception)
{
  if (file == (const FILE *) NULL)
    file=stdout;
  AcquireSemaphoreInfo(&resource_semaphore);
  (void) fprintf(file,"File    Memory       Map       Disk\n");
  (void) fprintf(file,"-----------------------------------\n");
  (void) fprintf(file,"%4lu  %6lumb  %6lumb  %6lugb\n",
    resource_info.file_limit,resource_info.memory_limit,resource_info.map_limit,
    resource_info.disk_limit/1024);
  (void) fflush(file);
  LiberateSemaphoreInfo(&resource_semaphore);
  return(True);
}
/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   S e t M a g i c k R e s o u r c e L i m i t                               %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  SetMagickResourceLimit() sets the limit for a particular resource in
%  megabytes.
%
%  The format of the SetMagickResourceLimit() method is:
%
%      void SetMagickResourceLimit(const ResourceType type,
%        const unsigned long limit)
%
%  A description of each parameter follows:
%
%    o type: The type of resource.
%
%    o limit: The maximum limit for the resource.
%
%
*/
MagickExport void SetMagickResourceLimit(const ResourceType type,
  const unsigned long limit)
{
  AcquireSemaphoreInfo(&resource_semaphore);
  switch (type)
  {
    case FileResource:
    {
      resource_info.file_limit=limit;
      break;
    }
    case MemoryResource:
    {
      resource_info.memory_limit=limit;
      break;
    }
    case MapResource:
    {
      resource_info.map_limit=limit;
      break;
    }
    case DiskResource:
    {
      resource_info.disk_limit=limit;
      break;
    }
    default:
      break;
  }
  LiberateSemaphoreInfo(&resource_semaphore);
}
